/**
 *      Much of the FTP code was inspired by the nanoftp.c module from
 *      libxml2 (Copyright Daniel Veillard, 2003).  The routines have been
 *      modified to fit the needs of the Motion project.
 *
 *      Copyright 2005, William M. Brack
 *      This software is distributed under the GNU Public license Version 2.
 *      See also the file 'COPYING'.
 *      
 */
#include "motion.h"  /* needs to come first, because _GNU_SOURCE_ set there */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <regex.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <netinet/in.h>

#include "netcam.h"
#include "netcam_ftp.h"

/**
 * ftpNewCtxt
 *
 *      Create a new FTP context structure
 *
 * Parameters
 * 
 *       None
 *
 * Returns:     Pointer to the newly-created structure, NULL if error
 *
 */
ftpCtxtPtr ftpNewCtxt(void)
{
    ftpCtxtPtr ret;
    
    /* note that mymalloc will exit on any problem */
    ret = mymalloc(sizeof(ftpCtxt));

    memset(ret, 0, sizeof(ftpCtxt));
    ret->controlFd = -1;                /* no control connection yet */
    ret->dataFd = -1;                   /* no data connection yet */
    return ret;
}

/**
 * ftpFreeCtxt
 *
 *      Free the resources allocated for this context
 *
 * Parameters
 * 
 *      ctxt    Pointer to the ftpCtxt structure
 *
 * Returns:     Nothing
 *
 */
void ftpFreeCtxt(ftpCtxtPtr ctxt)
{
    if (ctxt == NULL)
	return;
    if (ctxt->path != NULL)
	free(ctxt->path);
    if (ctxt->user)
	free(ctxt->user);
    if (ctxt->passwd)
	free(ctxt->passwd);
    if (ctxt->controlFd >= 0)
	close(ctxt->controlFd);
    free(ctxt);
}

/**
 * ftpParseResponse
 *
 *      Parses the answer from the server, extracting the numeric code.
 *
 * Parameters:
 * 
 *      buf     the buffer containing the response
 *      len     the buffer length
 * 
 * Returns:
 *     0 for errors
 *     +XXX for last line of response
 *     -XXX for response to be continued
 */
static int ftpParseResponse(char *buf, int len) {
    int val = 0;

    if (len < 3) return(-1);
    if ((*buf >= '0') && (*buf <= '9')) 
        val = val * 10 + (*buf - '0');
    else
        return(0);
    buf++;
    if ((*buf >= '0') && (*buf <= '9')) 
        val = val * 10 + (*buf - '0');
    else
        return(0);
    buf++;
    if ((*buf >= '0') && (*buf <= '9')) 
        val = val * 10 + (*buf - '0');
    else
        return(0);
    buf++;
    if (*buf == '-') 
        return(-val);
    return(val);
}

/**
 * ftpGetMore
 *
 *      Read more information from the FTP control connection
 *
 * Parameters:
 * 
 *      ctxt    pointer to an FTP context
 *      
 * Returns the number of bytes read, < 0 indicates an error
 */
static int ftpGetMore(ftpCtxtPtr ctxt) {
    int len;
    int size;

    /* Validate that our context structure is valid */
    if ((ctxt == NULL) || (ctxt->controlFd < 0))
	return(-1);

    if ((ctxt->controlBufIndex < 0) || 
	(ctxt->controlBufIndex > FTP_BUF_SIZE)) {
	return(-1);
    }

    if ((ctxt->controlBufUsed < 0) ||
	(ctxt->controlBufUsed > FTP_BUF_SIZE)) {
	return(-1);
    }
    if (ctxt->controlBufIndex > ctxt->controlBufUsed) {
	return(-1);
    }

    /*
     * First pack the control buffer
     */
    if (ctxt->controlBufIndex > 0) {
	memmove(&ctxt->controlBuf[0],
	        &ctxt->controlBuf[ctxt->controlBufIndex],
	        ctxt->controlBufUsed - ctxt->controlBufIndex);
	ctxt->controlBufUsed -= ctxt->controlBufIndex;
	ctxt->controlBufIndex = 0;
    }
    size = FTP_BUF_SIZE - ctxt->controlBufUsed;
    if (size == 0) {
	return(0);
    }

    /*
     * Read the amount left on the control connection
     */
    if ((len = recv(ctxt->controlFd,
	    &ctxt->controlBuf[ctxt->controlBufIndex],
	    size, 0)) < 0) {
	motion_log(LOG_ERR, 1, "recv failed in ftpGetMore");
	close(ctxt->controlFd);
	ctxt->controlFd = -1;
	return(-1);
    }
    ctxt->controlBufUsed += len;
    ctxt->controlBuf[ctxt->controlBufUsed] = 0;

    return(len);
}

/**
 * ftpGetResponse
 *
 *      Read the response from the FTP server after a command.
 *
 * Parameters
 * 
 *      ctxt    pointer to an FTP context
 *      
 * Returns the code number
 */
static int ftpGetResponse(ftpCtxtPtr ctxt) {
    char *ptr, *end;
    int len;
    int res = -1, cur = -1;

    if ((ctxt == NULL) || (ctxt->controlFd < 0))
	return(-1);

get_more:
    /*
     * Assumes everything up to controlBuf[controlBufIndex] has been read
     * and analyzed.
     */
    len = ftpGetMore(ctxt);
    if (len < 0) {
	return(-1);
    }
    if ((ctxt->controlBufUsed == 0) && (len == 0)) {
	return(-1);
    }
    ptr = &ctxt->controlBuf[ctxt->controlBufIndex];
    end = &ctxt->controlBuf[ctxt->controlBufUsed];

    while (ptr < end) {
	cur = ftpParseResponse(ptr, end - ptr);
	if (cur > 0) {
	    /*
	     * Successfully scanned the control code, skip
	     * till the end of the line, but keep the index to be
	     * able to analyze the result if needed.
	     */
	    res = cur;
	    ptr += 3;
	    ctxt->controlBufAnswer = ptr - ctxt->controlBuf;
	    while ((ptr < end) && (*ptr != '\n')) ptr++;
	    if (*ptr == '\n') ptr++;
	    if (*ptr == '\r') ptr++;
	    break;
	}
	while ((ptr < end) && (*ptr != '\n'))
	    ptr++;
	if (ptr >= end) {
	    ctxt->controlBufIndex = ctxt->controlBufUsed;
	    goto get_more;
	}
	if (*ptr != '\r') ptr++;
    }

    if (res < 0)
	goto get_more;
    ctxt->controlBufIndex = ptr - ctxt->controlBuf;
    return(res / 100);
}

/**
 * Send the user authentication
 */

static int ftpSendUser(ftpCtxtPtr ctxt) {
    char buf[200];
    int len;
    int res;

    if (ctxt->user == NULL)
	snprintf(buf, sizeof(buf), "USER anonymous\r\n");
    else
	snprintf(buf, sizeof(buf), "USER %s\r\n", ctxt->user);
    buf[sizeof(buf) - 1] = 0;
    len = strlen(buf);
    res = send(ctxt->controlFd, buf, len, 0);
    if (res < 0) {
	motion_log(LOG_ERR, 1, "send failed in ftpSendUser");
	return(res);
    }
    return(0);
}

/**
 * Send the password authentication
 */

static int ftpSendPasswd(ftpCtxtPtr ctxt) {
    char buf[200];
    int len;
    int res;

    if (ctxt->passwd == NULL)
	snprintf(buf, sizeof(buf), "PASS anonymous@\r\n");
    else
	snprintf(buf, sizeof(buf), "PASS %s\r\n", ctxt->passwd);
    buf[sizeof(buf) - 1] = 0;
    len = strlen(buf);
    res = send(ctxt->controlFd, buf, len, 0);
    if (res < 0) {
	motion_log(LOG_ERR, 1, "send failed in ftpSendPasswd");
	return(res);
    }
    return(0);
}

/**
 * ftpQuit
 *
 *      Send a QUIT command to the server
 *
 * Parameters:
 * 
 *      ctxt    pointer to an FTP context
 *      
 * Returns -1 in case of error, 0 otherwise
 */


static int ftpQuit(ftpCtxtPtr ctxt) {
    char buf[200];
    int len, res;

    if ((ctxt == NULL) || (ctxt->controlFd < 0))
	return(-1);

    snprintf(buf, sizeof(buf), "QUIT\r\n");
    len = strlen(buf);
    res = send(ctxt->controlFd, buf, len, 0);
    if (res < 0) {
	motion_log(LOG_ERR, 1, "send failed in ftpQuit");
	return(res);
    }
    return(0);
}

/**
 * ftpConnect
 *
 *      Tries to open a control connection
 *
 * Parameters:
 * 
 *      ctxt    an FTP context
 *      
 * Returns -1 in case of error, 0 otherwise
 */

int ftpConnect(netcam_context_ptr netcam) {
    ftpCtxtPtr ctxt;
    struct hostent *hp;
    int port;
    int res;
    int addrlen = sizeof (struct sockaddr_in);

    if (netcam == NULL)
	return -1;
    ctxt = netcam->ftp;
    if (ctxt == NULL)
	return(-1);
    if (netcam->connect_host == NULL)
	return(-1);

    /*
     * do the blocking DNS query.
     */
    port = netcam->connect_port;
    if (port == 0)
	port = 21;

    memset (&ctxt->ftpAddr, 0, sizeof(ctxt->ftpAddr));

    hp = gethostbyname (netcam->connect_host);
    if (hp == NULL) {
	motion_log(LOG_ERR, 1, "gethostbyname failed in ftpConnect");
	return (-1);
    }
    if ((unsigned int) hp->h_length >
        sizeof(((struct sockaddr_in *)&ctxt->ftpAddr)->sin_addr)) {
	motion_log(LOG_ERR, 1, "gethostbyname address mismatch "
	                       "in ftpConnect");
	return (-1);
    }

    /*
     * Prepare the socket
     */
    ((struct sockaddr_in *)&ctxt->ftpAddr)->sin_family = AF_INET;
    memcpy (&((struct sockaddr_in *)&ctxt->ftpAddr)->sin_addr,
            hp->h_addr_list[0], hp->h_length);
    ((struct sockaddr_in *)&ctxt->ftpAddr)->sin_port =
            (u_short)htons ((unsigned short)port);
    ctxt->controlFd = socket (AF_INET, SOCK_STREAM, 0);
    addrlen = sizeof (struct sockaddr_in);

    if (ctxt->controlFd < 0) {
	motion_log(LOG_ERR, 1, "socket failed");
	return(-1);
    }

    /*
     * Do the connect.
     */
    if (connect(ctxt->controlFd, (struct sockaddr *) &ctxt->ftpAddr,
        addrlen) < 0) {
	motion_log(LOG_ERR, 1, "Failed to create a connection");
	close(ctxt->controlFd); ctxt->controlFd = -1;
	ctxt->controlFd = -1;
	return(-1);
    }

    /*
     * Wait for the HELLO from the server.
     */
    res = ftpGetResponse(ctxt);
    if (res != 2) {
	close(ctxt->controlFd); ctxt->controlFd = -1;
	ctxt->controlFd = -1;
	return(-1);
    }

    /*
     * Do the authentication
     */
    res = ftpSendUser(ctxt);
    if (res < 0) {
	close(ctxt->controlFd); ctxt->controlFd = -1;
	ctxt->controlFd = -1;
	return(-1);
    }
    res = ftpGetResponse(ctxt);
    switch (res) {
	case 2:
	    return(0);
	case 3:
	    break;
	case 1:
	case 4:
	case 5:
	case -1:
	default:
	    close(ctxt->controlFd); ctxt->controlFd = -1;
	    ctxt->controlFd = -1;
	    return(-1);
    }
    res = ftpSendPasswd(ctxt);
    if (res < 0) {
	close(ctxt->controlFd); ctxt->controlFd = -1;
	ctxt->controlFd = -1;
	return(-1);
    }
    res = ftpGetResponse(ctxt);
    switch (res) {
	case 2:
	    break;
	case 3:
	    motion_log(LOG_ERR, 0, "FTP server asking for ACCT on anonymous");
	case 1:
	case 4:
	case 5:
	case -1:
	default:
	    close(ctxt->controlFd); ctxt->controlFd = -1;
	    ctxt->controlFd = -1;
	    return(-1);
    }

    return(0);
}

/**
 * ftpGetConnection
 *
 *      Try to open a data connection to the server.
 *
 * Parameters:
 * 
 *      ctxt    pointer to an FTP context
 *      
 * Returns -1 incase of error, 0 otherwise
 */

static int ftpGetConnection(ftpCtxtPtr ctxt) {
    char buf[200], *cur;
    int len, i;
    int res;
    int on;
    unsigned char ad[6], *adp, *portp;
    unsigned int temp[6];
    struct sockaddr_in dataAddr;
    unsigned int dataAddrLen;

    if (ctxt == NULL)
	return(-1);

    /* set up a socket for our data address */
    if (ctxt->dataFd != -1)
	close(ctxt->dataFd);
    memset (&dataAddr, 0, sizeof(dataAddr));
    ctxt->dataFd = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (ctxt->dataFd < 0) {
	motion_log(LOG_ERR, 1, "socket failed");
	return (-1);
    }
    on = 1;
    if (setsockopt(ctxt->dataFd, SOL_SOCKET, SO_REUSEADDR,
        (char *)&on, sizeof(on)) < 0) {
	motion_log(LOG_ERR, 1, "setting socket option SO_REUSEADDR");
	return -1;
    }

    ((struct sockaddr_in *)&dataAddr)->sin_family = AF_INET;
    dataAddrLen = sizeof (struct sockaddr_in);

    if (ctxt->passive) {
	/* send PASV command over control channel */
	snprintf (buf, sizeof(buf), "PASV\r\n");
	len = strlen (buf);
	res = send(ctxt->controlFd, buf, len, 0);
	if (res < 0) {
	    motion_log(LOG_ERR, 1, "send failed in ftpGetConnection");
	    close(ctxt->dataFd);
	    ctxt->dataFd = -1;
	    return(res);
	}
	/* check server's answer */
	res = ftpGetResponse(ctxt);
	if (res != 2) {
	    if (res == 5) {
		close(ctxt->dataFd); ctxt->dataFd = -1;
		return(-1);
	    } else {
		/*
		 * retry with an active connection
		 */
		close(ctxt->dataFd); ctxt->dataFd = -1;
		ctxt->passive = 0;
	    }
	}
	/* parse the IP address and port supplied by the server */
	cur = &ctxt->controlBuf[ctxt->controlBufAnswer];
	while (((*cur < '0') || (*cur > '9')) && *cur != '\0')
	    cur++;
	if (sscanf (cur, "%u,%u,%u,%u,%u,%u", &temp[0], &temp[1], &temp[2],
	            &temp[3], &temp[4], &temp[5]) != 6) {
	    motion_log(LOG_ERR, 0, "Invalid answer to PASV");
	    if (ctxt->dataFd != -1) {
		close (ctxt->dataFd); ctxt->dataFd = -1;
	    }
	    return (-1);
	}
	for (i=0; i<6; i++)
	    ad[i] = (unsigned char) (temp[i] & 0xff) ;
	memcpy (&((struct sockaddr_in *)&dataAddr)->sin_addr, &ad[0], 4);
	memcpy (&((struct sockaddr_in *)&dataAddr)->sin_port, &ad[4], 2);

	/* Now try to connect to the data port */
	if (connect(ctxt->dataFd, (struct sockaddr *) &dataAddr,
	            dataAddrLen) < 0) {
	    motion_log(LOG_ERR, 1, "Failed to create a data connection");
	    close(ctxt->dataFd); ctxt->dataFd = -1;
	    return (-1);
	}
    } else {
	/*
	 * We want to bind to a port to receive the data.  To do this,
	 * we need the address of our host.  One easy way to get it is
	 * to get the info from the control connection that we have
	 * with the remote server
	 */
	getsockname(ctxt->controlFd, (struct sockaddr *)&dataAddr,
	            &dataAddrLen);
	((struct sockaddr_in *)&dataAddr)->sin_port = 0;

	/* bind to the socket - should give us a unique port */
	if (bind(ctxt->dataFd, (struct sockaddr *) &dataAddr,
		 dataAddrLen) < 0) {
	    motion_log(LOG_ERR, 1, "bind failed");
	    close(ctxt->dataFd); ctxt->dataFd = -1;
	    return (-1);
	}

	/* we get the port number by reading back in the sockaddr */
	getsockname(ctxt->dataFd, (struct sockaddr *)&dataAddr,
	            &dataAddrLen);

	/* set up a 'listen' on the port to get the server's connection */
	if (listen(ctxt->dataFd, 1) < 0) {
	    motion_log(LOG_ERR, 1, "listen failed");
	    close(ctxt->dataFd); ctxt->dataFd = -1;
	    return (-1);
	}

	/* now generate the PORT command */
	adp = (unsigned char *) &((struct sockaddr_in *)&dataAddr)->sin_addr;
	portp = (unsigned char *) &((struct sockaddr_in *)&dataAddr)->sin_port;
	snprintf (buf, sizeof(buf), "PORT %d,%d,%d,%d,%d,%d\r\n",
	          adp[0] & 0xff, adp[1] & 0xff, adp[2] & 0xff, adp[3] & 0xff,
	          portp[0] & 0xff, portp[1] & 0xff);

        buf[sizeof(buf) - 1] = 0;
        len = strlen(buf);

	/* send the PORT command to the server */
	res = send(ctxt->controlFd, buf, len, 0);
	if (res < 0) {
	    motion_log(LOG_ERR, 1, "send failed in ftpGetConnection");
	    close(ctxt->dataFd);
	    ctxt->dataFd = -1;
	    return(res);
	}
	res = ftpGetResponse(ctxt);
	if (res != 2) {
	    close(ctxt->dataFd);
	    ctxt->dataFd = -1;
	    return(-1);
        }
    }
    return(ctxt->dataFd);
}

/**
 * ftpCloseConnection
 *
 *      Close the data connection from the server
 *
 * Parameters:
 * 
 *      ctxt    Pointer to an FTP context
 *      
 * Returns -1 in case of error, 0 otherwise
 */

static int ftpCloseConnection(ftpCtxtPtr ctxt) {
    int res;
    fd_set rfd, efd;
    struct timeval tv;

    if ((ctxt == NULL) || (ctxt->controlFd < 0)) return(-1);

    close(ctxt->dataFd);
    ctxt->dataFd = -1;
   
    /* Check for data on the control channel */
    tv.tv_sec = 15;
    tv.tv_usec = 0;
    FD_ZERO(&rfd);
    FD_SET(ctxt->controlFd, &rfd);
    FD_ZERO(&efd);
    FD_SET(ctxt->controlFd, &efd);
    res = select(ctxt->controlFd + 1, &rfd, NULL, &efd, &tv);
    if (res < 0) {
	close(ctxt->controlFd); ctxt->controlFd = -1;
	return(-1);
    }
    if (res == 0) {             /* timeout */
	close(ctxt->controlFd); ctxt->controlFd = -1;
    } else {                    /* read the response */
	res = ftpGetResponse(ctxt);
	if (res != 2) {            /* should be positive completion (2) */
	    close(ctxt->controlFd); ctxt->controlFd = -1;
	    return(-1);
	}
    }
    return(0);
}

/**
 * ftpGetSocket
 *
 *      Initiate fetch of the given file from the server.
 *
 * Parameters:
 * 
 *      ctxt            an FTP context
 *
 * Returns the socket for the data connection, or <0 in case of error
 */


int ftpGetSocket(ftpCtxtPtr ctxt) {
    char buf[300];
    int res, len;
    int acfd;
    
    if ((ctxt == NULL) || (ctxt->path == NULL))
	return(-1);

    /* Set up the data connection */
    ctxt->dataFd = ftpGetConnection(ctxt);
    if (ctxt->dataFd == -1)
	return(-1);

    /* generate a "retrieve" command for the file */
    snprintf(buf, sizeof(buf), "RETR %s\r\n", ctxt->path);
    buf[sizeof(buf) - 1] = 0;
    len = strlen(buf);
    
    /* send it to the server */
    res = send(ctxt->controlFd, buf, len, 0);
    if (res < 0) {
	motion_log(LOG_ERR, 1, "send failed in ftpGetSocket");
	close(ctxt->dataFd);
	ctxt->dataFd = -1;
	return(res);
    }
    
    /* check the answer */
    res = ftpGetResponse(ctxt);
    if (res != 1) {
	close(ctxt->dataFd); ctxt->dataFd = -1;
	return(-res);
    }

    /*
     * if not a passive connection, need to do an accept to get the
     * connection from the server
     */
    if (!ctxt->passive) {
	struct sockaddr_in dataAddr;
	unsigned int dataAddrLen = sizeof(struct sockaddr_in);
	if ((acfd = accept(ctxt->dataFd, (struct sockaddr *)&dataAddr,
	     &dataAddrLen)) < 0) {
	    motion_log(LOG_ERR, 1, "accept in ftpGetSocket");
	    close(ctxt->dataFd);
	    ctxt->dataFd = -1;
	    return -1;
	}
	close(ctxt->dataFd);
	ctxt->dataFd = acfd;
    }

    return(ctxt->dataFd);
}
/**
 * ftpSendType
 *
 * 	Send a TYPE (either 'I' or 'A') command to the server
 *
 * Parameters
 *
 *      ctxt    pointer to the ftpCtxt
 *      type    ascii character ('I' or 'A')
 *
 * Returns      0 for success, negative error code for failure
 *
 */
int ftpSendType(ftpCtxtPtr ctxt, char type) {
    char buf[100], utype;
    int len, res;
    
    utype = toupper(type);
    /* Assure transfer will be in "image" mode */
    snprintf(buf, sizeof(buf), "TYPE I\r\n");
    len = strlen(buf);
    res = send(ctxt->controlFd, buf, len, 0);
    if (res < 0) {
	motion_log(LOG_ERR, 1, "send failed in ftpGetSocket");
	close(ctxt->dataFd); ctxt->dataFd = -1;
	return(res);
    }
    res = ftpGetResponse(ctxt);
    if (res != 2) {
	close(ctxt->dataFd); ctxt->dataFd = -1;
	return(-res);
    }
    return 0;
}

/**
 * ftpRead
 *
 *      This function tries to read len bytes from the existing FTP
 *      connection and saves them in dest. This is a blocking call.
 * 
 * Parameters:
 *      ctxt    the FTP context
 *      dest    a buffer
 *      len     the buffer length
 *      
 * Returns:     the number of bytes read.
 *              0 is an indication of an end of connection.
 *              -1 indicates a parameter error.
 */
int ftpRead(ftpCtxtPtr ctxt, void *dest, int len) {

    if (ctxt == NULL)
	return(-1);
    if (ctxt->dataFd < 0)
	return(0);
    if (dest == NULL)
	return(-1);
    if (len <= 0)
	return(0);

    len = recv(ctxt->dataFd, dest, len, 0);
    if (len <= 0) {
	if (len < 0)
	    motion_log(LOG_ERR, 1, "recv failed in ftpRead");
	ftpCloseConnection(ctxt);
    }
    return(len);
}

/**
 * ftpClose
 *
 *      Close the connection and both control and transport
 *
 * Parameters:
 *
 *      ctxt    Pointer to an FTP context
 *      
 * Returns -1 in case of error, 0 otherwise
 */

int ftpClose(ftpCtxtPtr ctxt) {

    if (ctxt == NULL)
	return(-1);

    if (ctxt->dataFd >= 0) {
	close(ctxt->dataFd);
	ctxt->dataFd = -1;
    }
    if (ctxt->controlFd >= 0) {
	ftpQuit(ctxt);
	close(ctxt->controlFd);
	ctxt->controlFd = -1;
    }
    ftpFreeCtxt(ctxt);
    return(0);
}
