/*
 *	webcam.c
 *	Streaming webcam using jpeg images over a multipart/x-mixed-replace stream
 *	Copyright (C) 2002 Jeroen Vreeken (pe1rxq@amsat.org)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <sys/fcntl.h>
#include <sys/stat.h>

#include "picture.h"
#include "webcam.h"
#include "motion.h"


/* This function sets up a TCP/IP socket for incomming requests. It is called only during
 * initiatization of Motion from the function webcam_init
 * The function sets up a a socket on the port number given by _port_.
 * If the parameter _local_ is not zero the socket is setup to only accept connects from localhost.
 * Otherwise any client IP address is accepted. The function returns an integer representing the socket.
 */
int http_bindsock(int port, int local, struct context *cnt)
{
	int sl, optval=1;
	struct sockaddr_in sin;

	if ((sl=socket(PF_INET, SOCK_STREAM, 0))<0) {
		printf("[%d] socket(): %m\n",cnt->threadnr);
		syslog(LOG_ERR, "[%d] socket(): %m",cnt->threadnr);
		return -1;
	}

	memset(&sin, 0, sizeof(struct sockaddr_in));
	sin.sin_family=AF_INET;
	sin.sin_port=htons(port);
	if (local)
		sin.sin_addr.s_addr=htonl(INADDR_LOOPBACK);
	else
		sin.sin_addr.s_addr=htonl(INADDR_ANY);

	setsockopt(sl, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));

	if (bind(sl, (struct sockaddr *)&sin, sizeof(struct sockaddr_in))==-1) {
		printf("[%d] bind(): %m\n",cnt->threadnr);
		syslog(LOG_ERR, "[%d] bind(): %m",cnt->threadnr);
		close(sl);
		return -1;
	}

	if (listen(sl, DEF_MAXWEBQUEUE)==-1) {
		printf("[%d] listen(): %m\n",cnt->threadnr);
		syslog(LOG_ERR, "[%d] listen(): %m",cnt->threadnr);
		close(sl);
		return -1;
	}

	return sl;
}


int http_acceptsock(int sl, struct context *cnt)
{
	int sc, i;
	struct sockaddr_in sin;
	socklen_t addrlen=sizeof(struct sockaddr_in);

	if ((sc=accept(sl, (struct sockaddr *)&sin, &addrlen))>=0) {
		i=1;
		ioctl(sc, FIONBIO, &i);
		return sc;
	}
	printf("[%d] accept(): %m\n",cnt->threadnr);
	syslog(LOG_ERR, "[%d] accept(): %m",cnt->threadnr);

	return -1;
}


/* Webcam flush sends the current jpeg frame to all connected clients */
void webcam_flush(struct webcam *list, int *stream_count, int lim)
{
	int written;
	void *tmp;

	while (list->next) {
		list=list->next;
		if (!list->tmpbuffer)
			continue;
		
		while (list->tmpbuffer && list->filepos < list->tmpbuffer->size) {
			written=write(list->socket, list->tmpbuffer->ptr, list->tmpbuffer->size);
			
			/* Test that we have successfully written at least one byte to socket */
			if (written>0)
				list->filepos+=written;
			
			/* Test that we have written the entire buffer to the socket
			 * and no error message. THEN WHAT???
			 * and finally free the memory buffer area again.
			 */
			if (list->filepos >= list->tmpbuffer->size || (written < 0 && errno!=EAGAIN)) {
				if (--list->tmpbuffer->ref <= 0) {
					free(list->tmpbuffer->ptr);
					free(list->tmpbuffer);
				}
				
				list->tmpbuffer=NULL;
				list->nr++;
			}
			
			/* Disconnect client if he is no longer connected and free the webcam struct */
			if ( (written<0 && errno!=EAGAIN) || (lim && !list->tmpbuffer && list->nr>lim) ) {
				fclose(list->fwrite);
				close(list->socket);
				if (list->next)
					list->next->prev=list->prev;
				list->prev->next=list->next;
				tmp=list;
				list=list->prev;
				free(tmp);
				(*stream_count)--;
			}
		}
	}
}


struct webcam_buffer *webcam_tmpbuffer(int size)
{
	struct webcam_buffer *tmpbuffer=mymalloc(sizeof(struct webcam_buffer));
	tmpbuffer->ref=0;
	tmpbuffer->ptr=mymalloc(size);
	if (!tmpbuffer->ptr) {
		free(tmpbuffer);
		return NULL;
	}
	return tmpbuffer;
}


void webcam_add_client(struct webcam *list, int sc)
{
	struct webcam *new=mymalloc(sizeof(struct webcam));
	new->socket=sc;
	new->fwrite=fdopen(sc, "a");
	new->tmpbuffer=NULL;
	new->filepos=0;
	new->nr=0;
	new->prev=list;
	new->next=list->next;
	if (new->next)
		new->next->prev=new;
	list->next=new;
	new->last=0;

	fprintf(new->fwrite, "HTTP/1.0 200 OK\r\n");
	fprintf(new->fwrite, "Server: Motion/"VERSION"\r\n");
	fprintf(new->fwrite, "Connection: close\r\n");
	fprintf(new->fwrite, "Max-Age: 0\r\n");
	fprintf(new->fwrite, "Expires: 0\r\n");
	fprintf(new->fwrite, "Cache-Control: no-cache, private\r\n");
	fprintf(new->fwrite, "Pragma: no-cache\r\n");
	fprintf(new->fwrite, "Content-Type: multipart/x-mixed-replace; boundary=--BoundaryString\r\n\r\n");
	fflush(new->fwrite);
}


void webcam_add_write(struct webcam *list, struct webcam_buffer *tmpbuffer, int fps)
{
	struct timeval curtimeval;
	unsigned long int curtime;

	gettimeofday(&curtimeval, NULL);
	curtime=curtimeval.tv_usec+1000000L*curtimeval.tv_sec;
	while (list->next) {
		list=list->next;
		if (list->tmpbuffer==NULL && (curtime-list->last)>=1000000L/fps) {
			list->last=curtime;
			list->tmpbuffer=tmpbuffer;
			tmpbuffer->ref++;
			list->filepos=0;
			fprintf(list->fwrite, "--BoundaryString\r\n");
			fprintf(list->fwrite, "Content-type: image/jpeg\r\n");
			fprintf(list->fwrite, "Content-Length: %ld\r\n\r\n", list->tmpbuffer->size-2);
			fflush(list->fwrite);
		}
	}
	if (tmpbuffer->ref<=0) {
		free(tmpbuffer->ptr);
		free(tmpbuffer);
	}
}


/* We walk through the chain of webcam structs until we reach the end.
 * Here we check if the tmpbuffer points to NULL
 * We return 1 if it finds a list->tmpbuffer which is a NULL pointer which would
 * be the next client ready to be sent a new image. If not a 0 is returned.
 */
int webcam_check_write(struct webcam *list)
{
	while (list->next) {
		list=list->next;
		if (list->tmpbuffer==NULL)
			return 1;
	}
	return 0;
}


/* This function is called from motion.c for each motion thread starting up.
 * The function setup the incomming tcp socket that the clients connect to
 * The function returns an integer representing the socket.
 */
int webcam_init(struct context *cnt)
{
	cnt->webcam.socket=http_bindsock(cnt->conf.webcam_port, cnt->conf.webcam_localhost, cnt);
	cnt->webcam.next=NULL;
	cnt->webcam.prev=NULL;
	signal(SIGPIPE, SIG_IGN);
	return cnt->webcam.socket;
}

/* This function is called from the motion_loop when it ends
 * and motion is terminated or restarted
 */
void webcam_stop(struct context *cnt)
{	
	struct webcam *list;
	struct webcam *next = cnt->webcam.next;

	if (cnt->conf.setup_mode)
		printf("[%d] Closing webcam listen socket\n", cnt->threadnr);
	close(cnt->webcam.socket);
	if (cnt->conf.setup_mode)
		printf("[%d] Closing active webcam sockets\n", cnt->threadnr);

	while (next) {
		list=next;
		next=list->next;
		if (list->tmpbuffer) {
			free(list->tmpbuffer->ptr);
			free(list->tmpbuffer);
		}
		fclose(list->fwrite);
		close(list->socket);
		free(list);
	}
}

/* webcam_put is the starting point of the webcam loop. It is called from the motion_loop
 * If config option 'webcam_motion' is 'on' this function is called once per second (frame 0)
 * and when Motion is detected excl pre_capture.
 * If config option 'webcam_motion' is 'off' this function is called once per captured
 * picture frame.
 * It is always run in setup mode for each picture frame captured and with the special
 * setup image.
 * The function has two functions:
 * It looks for possible waiting new clients and adds them.
 * It sends latest picture frame to all connected clients.
 * Note: Clients that have disconnected are handled in the webcam_flush() function
 */
int webcam_put(struct context *cnt, unsigned char *image)
{
	struct timeval timeout; 
	struct webcam_buffer *tmpbuffer;
	fd_set fdread;
	int sl=cnt->webcam.socket;
	int sc;

	/* timeout struct used to timeout the time we wait for a client and we do not wait at all */
	timeout.tv_sec=0;
	timeout.tv_usec=0;
	FD_ZERO(&fdread);
	FD_SET(cnt->webcam.socket, &fdread);
	
	
	/* If we have not reached the max number of allowed clients per thread
	 * we will check to see if new clients are waiting to connect
	 * If this is the case we add the client as a new webcam struct and add this to
	 * the end of the chain of webcam structs that are linked to each other.
	 */
	if ((cnt->stream_count < DEF_MAXSTREAMS) &&
	    (select(sl+1, &fdread, NULL, NULL, &timeout)>0)) {
		sc=http_acceptsock(sl, cnt);
		webcam_add_client(&cnt->webcam, sc);
		cnt->stream_count++;
	}
	
	webcam_flush(&cnt->webcam, &cnt->stream_count, cnt->conf.webcam_limit);
	
	if (webcam_check_write(&cnt->webcam)) {
		tmpbuffer=webcam_tmpbuffer(cnt->imgs.size);
		if (tmpbuffer) {
			tmpbuffer->size=put_picture_memory(cnt, &tmpbuffer->ptr, cnt->imgs.size, image, cnt->conf.webcam_quality);
			memcpy(tmpbuffer->ptr + tmpbuffer->size, "\r\n", 2);
			tmpbuffer->size += 2;
			webcam_add_write(&cnt->webcam, tmpbuffer, cnt->conf.webcam_maxrate);
		} else {
			const char msg[]="Error creating tmpbuffer: ";
			printf("[%d] %s%m\n", cnt->threadnr, msg);
			syslog(LOG_ERR, "[%d] %s%m", cnt->threadnr, msg);
		}
	}
	
	webcam_flush(&cnt->webcam, &cnt->stream_count, cnt->conf.webcam_limit);
	
	return 0;
}
