/*	picture.c
 *
 *	Various funtions for saving/loading pictures.
 *	Copyright 2002 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	Portions of this file are Copyright by Lionnel Maugis
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "picture.h"
#include "event.h"

#include <jpeglib.h>


// twice as fast as RGB jpeg 
void put_jpeg_yuv420p (FILE *fp, unsigned char *image, int width, int height, int quality)
{
	int i,j;

	JSAMPROW y[16],cb[16],cr[16]; // y[2][5] = color sample of row 2 and pixel column 5; (one plane)
	JSAMPARRAY data[3]; // t[0][2][5] = color sample 0 of row 2 and column 5

	struct jpeg_compress_struct cinfo;
	struct jpeg_error_mgr jerr;

	data[0] = y;
	data[1] = cb;
	data[2] = cr;

	cinfo.err = jpeg_std_error(&jerr);  // errors get written to stderr 
	
	jpeg_create_compress (&cinfo);
	cinfo.image_width = width;
	cinfo.image_height = height;
	cinfo.input_components = 3;
	jpeg_set_defaults (&cinfo);

	jpeg_set_colorspace(&cinfo, JCS_YCbCr);

	cinfo.raw_data_in = TRUE; // supply downsampled data
	cinfo.comp_info[0].h_samp_factor = 2;
	cinfo.comp_info[0].v_samp_factor = 2;
	cinfo.comp_info[1].h_samp_factor = 1;
	cinfo.comp_info[1].v_samp_factor = 1;
	cinfo.comp_info[2].h_samp_factor = 1;
	cinfo.comp_info[2].v_samp_factor = 1;

	jpeg_set_quality (&cinfo, quality, TRUE);
	cinfo.dct_method = JDCT_FASTEST;

	jpeg_stdio_dest (&cinfo, fp);  	  // data written to file
	jpeg_start_compress (&cinfo, TRUE);

	for (j=0;j<height;j+=16) {
		for (i=0;i<16;i++) {
			y[i] = image + width*(i+j);
			if (i%2 == 0) {
				cb[i/2] = image + width*height + width/2*((i+j)/2);
				cr[i/2] = image + width*height + width*height/4 + width/2*((i+j)/2);
			}
		}
		jpeg_write_raw_data (&cinfo, data, 16);
	}

	jpeg_finish_compress (&cinfo);
	jpeg_destroy_compress (&cinfo);
}

void put_jpeg_grey (FILE *picture, char *image, int width, int height, int quality)
{
	int y;
	JSAMPROW row_ptr[1];
	struct jpeg_compress_struct cjpeg;
	struct jpeg_error_mgr jerr;

	cjpeg.err = jpeg_std_error(&jerr);
	jpeg_create_compress (&cjpeg);
	cjpeg.image_width = width;
	cjpeg.image_height = height;
	cjpeg.input_components = 1; /* one colour component */
	cjpeg.in_color_space = JCS_GRAYSCALE;

	jpeg_set_defaults (&cjpeg);

	jpeg_set_quality (&cjpeg, quality, TRUE);
	cjpeg.dct_method = JDCT_FASTEST;
	jpeg_stdio_dest (&cjpeg, picture);

	jpeg_start_compress (&cjpeg, TRUE);

	row_ptr[0]=image;
	for (y=0; y<height; y++) {
		jpeg_write_scanlines(&cjpeg, row_ptr, 1);
		row_ptr[0]+=width;
	}
	jpeg_finish_compress(&cjpeg);
	jpeg_destroy_compress(&cjpeg);
}

void put_ppm_bgr24 (FILE *picture, char *image, int width, int height)
{
	int x, y;
	unsigned char *l=image;
	unsigned char *u=image+width*height;
	unsigned char *v=u+(width*height)/4;
	int r, g, b;
	unsigned char rc, gc, bc;

	/*	ppm header
	 *	width height
	 *	maxval
	 */
	fprintf(picture, "P6\n");
	fprintf(picture, "%d %d\n", width, height);
	fprintf(picture, "%d\n", 255);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			r = 76283*(((int)*l)-16)+104595*(((int)*u)-128);
			g = 76283*(((int)*l)-16)- 53281*(((int)*u)-128)-25625*(((int)*v)-128);
			b = 76283*(((int)*l)-16)+132252*(((int)*v)-128);
			r = r>>16;
			g = g>>16;
			b = b>>16;
			if (r<0)
				r=0;
			else if (r>255)
				r=255;
			if (g<0)
				g=0;
			else if (g>255)
				g=255;
			if (b<0)
				b=0;
			else if (b>255)
				b=255;
			rc=r;
			gc=g;
			bc=b;

			l++;
			if (x&1) {
				u++;
				v++;
			}
			/* ppm is rgb not bgr */
			fwrite(&bc, 1, 1, picture);
			fwrite(&gc, 1, 1, picture);
			fwrite(&rc, 1, 1, picture);
		}
		if (y&1) {
			u-=width/2;
			v-=width/2;
		}
	}
}

/* copy smartmask as an overlay into motion images and movies */
void overlay_smartmask (struct context *cnt, unsigned char *out)
{
	int i, x, v, width, height, line;
	struct images *imgs=&cnt->imgs;
	unsigned char *smartmask=imgs->smartmask_final;
	
	i=imgs->motionsize;
	v=i+((imgs->motionsize)/4);
	width=imgs->width;
	height=imgs->height;

	/* set V to 255 to make smartmask appear red */
	out+=v;
	for ( i=0; i<height; i+=2){
		line=i*width;
		for (x=0; x<width; x+=2){
			if (smartmask[line+x]==0 ||
				smartmask[line+x+1]==0 ||
				smartmask[line+width+x]==0 ||
				smartmask[line+width+x+1]==0){
					*out=255;
			}
			out++;
		}
	}
}

/* copy fixed mask as an overlay into motion images and movies */
void overlay_fixed_mask (struct context *cnt, unsigned char *out)
{
	int i, x, v, width, height, line;
	struct images *imgs=&cnt->imgs;
	unsigned char *mask=imgs->mask;
	unsigned char *out_back=out;
	
	i=imgs->motionsize;
	v=i+((imgs->motionsize)/4);
	width=imgs->width;
	height=imgs->height;

	/* set V to 255 to make fixed mask appear red */
	out+=v;
	for ( i=0; i<height; i+=2){
		line=i*width;
		for (x=0; x<width; x+=2){
			if (mask[line+x]<255 ||
				mask[line+x+1]<255 ||
				mask[line+width+x]<255 ||
				mask[line+width+x+1]<255){
					*out=255;
			}
			out++;
		}
	}
	out=out_back;
	/* set colour intensity for fixed mask to reflect mask sensitivity */
	for (i=0; i<imgs->motionsize; i++){
		if (mask[i]<255)
			*out=mask[i];
		*out++;
	}
}

/* copy largest label as an overlay into motion images and movies */
void overlay_largest_label (struct context *cnt, unsigned char *out)
{
	int i, x, width, height, line;
	struct images *imgs=&cnt->imgs;
	int *labels=imgs->labels;
	register int largest_label=imgs->largest_label;
	unsigned char *out_back=out;
	
	i=imgs->motionsize;
	width=imgs->width;
	height=imgs->height;

	/* set U to 255 to make label appear blue */
	out+=i;
	for ( i=0; i<height; i+=2){
		line=i*width;
		for (x=0; x<width; x+=2){
			if (labels[line+x]==largest_label ||
				labels[line+x+1]==largest_label ||
				labels[line+width+x]==largest_label ||
				labels[line+width+x+1]==largest_label){
					*out=255;
			}
			out++;
		}
	}
	out=out_back;
	/* set intensity for coloured label to have better visibility */
	for (i=0; i<imgs->motionsize; i++){
		if (*labels++==largest_label)
			*out=0;
		out++;
	}
}

void put_picture_fd (struct context *cnt, FILE *picture, char *image, int quality)
{
	if (cnt->conf.ppm) {
		put_ppm_bgr24(picture, image, cnt->imgs.width, cnt->imgs.height);
	} else {
//		printf("type: %d %d\n", cnt->imgs.type, VIDEO_PALETTE_YUV420P);
		switch (cnt->imgs.type) {
			case VIDEO_PALETTE_YUV420P:
				put_jpeg_yuv420p(picture, image, cnt->imgs.width, cnt->imgs.height, quality);
				break;
			case VIDEO_PALETTE_GREY:
				put_jpeg_grey(picture, image, cnt->imgs.width, cnt->imgs.height, quality);
				break;
		}
	}
}

void put_picture (struct context *cnt, char *file, char *image, int ftype)
{
	FILE *picture;

	picture=myfopen(file, "w");
	if (!picture) {
		/* Report to syslog - suggest solution if the problem is access rights to target dir */
		if (errno ==  EACCES){
			syslog(LOG_ERR, "[%d] Can't write picture to file %s - check access rights to target directory: %m", cnt->threadnr, file);
			printf("[%d] Can't write picture to file %s - check access rights to target directory: %m\n", cnt->threadnr, file);
			cnt->finish = 1;
			return;
		} else {
			/* If target dir is temporarily unavailable we may survive */
			const char msg[]="Can't write picture to file ";
			printf("[%d] %s%s: %m\n", cnt->threadnr, msg, file);
			syslog(LOG_ERR, "[%d] %s%s: %m", cnt->threadnr, msg, file);
			return;
		}
	}

	put_picture_fd(cnt, picture, image, cnt->conf.quality);
	fclose(picture);
	event(EVENT_FILECREATE, cnt, file, (void *)(unsigned long)ftype,NULL);
}

char *get_pgm (struct context *cnt, FILE *picture, int width, int height)
{
	int x=0 ,y=0, maxval;
	char line[256];
	char *image;

	line[255]=0;
	fgets(line, 255, picture);
	if (strncmp(line, "P5", 2)) {
		const char msg[]="This is not a ppm file, starts with: ";
		printf("[%d] %s'%s'\n", cnt->threadnr, msg, line);
		syslog(LOG_ERR, "[%d] %s'%s'", cnt->threadnr, msg, line);
		return NULL;
	}
	/* skip comment */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	/* check size */
	sscanf(line, "%d %d", &x, &y);
	if (x!=width || y!=height) {
		const char msg1[]="Wrong image size: ";
		const char msg2[]=" should be ";
		printf("[%d] %s%dx%d%s%dx%d\n", cnt->threadnr, msg1, x, y, msg2, width, height);
		syslog(LOG_ERR, "[%d] %s%dx%d%s%dx%d", cnt->threadnr, msg1, x, y, msg2, width, height);
		return NULL;
	}
	/* Maximum value */
	line[0]='#';
	while (line[0]=='#') fgets(line, 255, picture);
	sscanf(line, "%d", &maxval);
	/* read data */
	image=mymalloc(width*height);
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			fread(&image[y*width+x], 1, 1, picture);
			image[y*width+x]=(int)image[y*width+x]*255/maxval;
		}
	}
	return image;
}

void put_pgm (FILE *picture, unsigned char *image, int width, int height)
{
	/* Write pgm-header */
        fprintf(picture, "P5\n");
        fprintf(picture, "%d %d\n", width, height);
        fprintf(picture, "%d\n", 255);
	/* write image data at once */
	fwrite(image, width, height, picture);
}

void put_fixed_mask (struct context *cnt, char *file)
{
	FILE *picture;

	picture=myfopen(file, "w");
	if (!picture) {
		/* Report to syslog - suggest solution if the problem is access rights to target dir */
		if (errno ==  EACCES){
			printf("[%d] can't write mask file %s - check access rights to target directory: %m\n", cnt->threadnr, file);
			syslog(LOG_ERR, "[%d] can't write mask file %s - check access rights to target directory: %m", cnt->threadnr, file);
		} else {
			/* If target dir is temporarily unavailable we may survive */
			printf("[%d] can't write mask file %s: %m\n", cnt->threadnr, file);
			syslog(LOG_ERR, "[%d] can't write mask file %s: %m", cnt->threadnr, file);
		}
		return;
	}
	memset(cnt->imgs.out, 255, cnt->imgs.motionsize); /* initialize to unset */
	put_pgm(picture, cnt->imgs.out, cnt->conf.width, cnt->conf.height);
	fclose(picture);
	{
	const char msg1[]="Creating empty mask ";
	const char msg2[]="Please edit this file and re-run motion to enable mask feature.";
	printf("[%d] %s%s.\n", cnt->threadnr, msg1, cnt->conf.mask_file);
	syslog(LOG_ERR, "[%d] %s%s.", cnt->threadnr, msg1, cnt->conf.mask_file);
	printf("[%d] %s\n", cnt->threadnr, msg2);
	syslog(LOG_ERR, "[%d] %s", cnt->threadnr, msg2);
	}
}

/* save 'best' preview_shot */
void preview_best(struct context *cnt)
{
#ifdef HAVE_FFMPEG
	int use_jpegpath;
#endif /* HAVE_FFMPEG */
	char *jpegpath;
	char previewname[PATH_MAX];
	char filename[PATH_MAX];

	if(cnt->preview_max){
#ifdef HAVE_FFMPEG
		/* Use filename of movie i.o. jpeg_filename when set to 'preview' */
		use_jpegpath=strcmp(cnt->conf.jpegpath, "preview");
	
		if (cnt->ffmpeg_new && !use_jpegpath){
			/* Replace avi/mpg with jpg/ppm and keep the rest of the filename */
			strncpy(previewname, cnt->newfilename, strlen(cnt->newfilename)-3);
			strcat(previewname, imageext(cnt));
			put_picture(cnt, previewname, cnt->imgs.preview_buffer , FTYPE_IMAGE);
			return;
		}
#endif /* HAVE_FFMPEG */
		/* Save best preview-shot also when no movies are recorded or jpegpath
		   is used. Filename has to be generated - nothing available to reuse! */
		//printf("preview_shot: different filename or picture only!\n");

		/* conf.jpegpath would normally be defined but if someone deleted it by control interface
		   it is better to revert to the default than fail */
		if (cnt->conf.jpegpath)
			jpegpath = cnt->conf.jpegpath;
		else
			jpegpath = DEF_JPEGPATH;
			
		mystrftime(filename, sizeof(filename), jpegpath, cnt->currenttime, cnt->event_nr, cnt->shots);
		sprintf(previewname, "%s/%s.%s", cnt->conf.filepath, filename, imageext(cnt));
		put_picture(cnt, previewname, cnt->imgs.preview_buffer , FTYPE_IMAGE);
	}
}
