/*
 *
 * GSturl
 * Copyright (C) 2005 Vincenzo Di Massa <hawk.it@tiscali.it>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/gst.h>

/* include this header if you want to use dynamic parameters
#include <gst/control/control.h>
*/
#include <curl/curl.h>
#include <curl/types.h>
#include <curl/easy.h>
#include <time.h>

#include "gsturlsrc.h"

/* Plugin signals and args */
enum {
  /* FILL ME */
  LAST_SIGNAL
};

enum {
  ARG_0,
  ARG_URL
};

static GstStaticPadTemplate src_factory =
GST_STATIC_PAD_TEMPLATE (
  "src",
  GST_PAD_SRC,
  GST_PAD_ALWAYS,
  GST_STATIC_CAPS_ANY
);

static void	gst_urlsrc_class_init	(GstUrlsrcClass *klass);
static void	gst_urlsrc_base_init	(GstUrlsrcClass *klass);
static void	gst_urlsrc_init	(GstUrlsrc *filter);

static void	gst_urlsrc_set_property(GObject *object, guint prop_id,
                                                 const GValue *value,
					         GParamSpec *pspec);
static void	gst_urlsrc_get_property(GObject *object, guint prop_id,
                                                 GValue *value,
						 GParamSpec *pspec);

static GstElementStateReturn gst_urlsrc_change_state (
						GstElement *element);

size_t 		WriteMemoryCallback(void *ptr, size_t size, 
					size_t nmemb, void  *data);

static GstData * gst_urlsrc_get (GstPad *pad);

static GstElementClass *parent_class = NULL;


GType
gst_gst_urlsrc_get_type (void)
{
  static GType plugin_type = 0;

  if (!plugin_type)
  {
    static const GTypeInfo plugin_info =
    {
      sizeof (GstUrlsrcClass),
      (GBaseInitFunc) gst_urlsrc_base_init,
      NULL,
      (GClassInitFunc) gst_urlsrc_class_init,
      NULL,
      NULL,
      sizeof (GstUrlsrc),
      0,
      (GInstanceInitFunc) gst_urlsrc_init,
    };
    plugin_type = g_type_register_static (GST_TYPE_ELEMENT,
	                                  "GstUrlsrc",
	                                  &plugin_info, 0);
  }
  return plugin_type;
}

static void
gst_urlsrc_base_init (GstUrlsrcClass *klass)
{
  static GstElementDetails plugin_details = {
    "URL source element",
    "Source/Video",
    "URL source element made using libcurl",
    "Vincenzo Di Massa <hawk.it@tiscali.it>"
  };
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gst_element_class_add_pad_template (element_class,
	gst_static_pad_template_get (&src_factory));
  gst_element_class_set_details (element_class, &plugin_details);
}

/* initialize the plugin's class */
static void
gst_urlsrc_class_init (GstUrlsrcClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;

  gobject_class = (GObjectClass*) klass;
  gstelement_class = (GstElementClass*) klass;

  parent_class = g_type_class_ref (GST_TYPE_ELEMENT);

  g_object_class_install_property (gobject_class, ARG_URL,
    g_param_spec_string ("url", "Url", "URL",
                          "http://dlink/IMAGE.JPG" , G_PARAM_READWRITE));

  gobject_class->set_property = gst_urlsrc_set_property;
  gobject_class->get_property = gst_urlsrc_get_property;
  gstelement_class->change_state = gst_urlsrc_change_state;
}

/* initialize the new element
 * instantiate pads and add them to element
 * set functions
 * initialize structure
 */
static void
gst_urlsrc_init (GstUrlsrc *src)
{
  GstElementClass *klass = GST_ELEMENT_GET_CLASS (src);

  
  src->srcpad = gst_pad_new_from_template (
	gst_element_class_get_pad_template (klass, "src"), "src");
  
  gst_element_add_pad (GST_ELEMENT (src), src->srcpad);
  gst_pad_set_get_function (src->srcpad, gst_urlsrc_get);

  src->url = "http://dlink/IMAGE.JPG";
  src->curl=0;
  GST_FLAG_UNSET(src,GST_URLSRC_OPEN);
  src->need_discont = 2;
  src->need_flush=TRUE;
  src->sent_bytes=0;
}

size_t WriteMemoryCallback(void *ptr, size_t size, size_t nmemb, void 
 *datain) 
{ 
     guint realsize; 
     GstBuffer *buf;
     guint8 *data;
     
     realsize = size * nmemb;
     buf = GST_BUFFER(datain); 
     
     if (realsize)
     {
        if  ( GST_BUFFER_MAXSIZE(buf) < GST_BUFFER_SIZE(buf) + 
			realsize + 1) 
	{
		while (GST_BUFFER_MAXSIZE(buf) < GST_BUFFER_SIZE(buf) + 
				realsize + 1) 
			(GST_BUFFER_MAXSIZE(buf)) *= 2;
		GST_BUFFER_DATA(buf) = (guint8 *) realloc( 
				GST_BUFFER_DATA(buf), 
				GST_BUFFER_MAXSIZE(buf));
	}
	data = GST_BUFFER_DATA(buf);
	   //printf("Data! size=%d realsize=%d \n", 
	   //GST_INFO_OBJECT(buf , \
			   GST_BUFFER_SIZE(buf) , realsize);
	   //fflush(stdout);
	   if (data ) { 
		   memcpy(&(data[GST_BUFFER_SIZE(buf)]), ptr, realsize); 
		   GST_BUFFER_SIZE(buf) += realsize;
		   data[GST_BUFFER_SIZE(buf)] = 0; 
	   } 
     }
     return realsize; 
}

static gboolean gst_urlsrc_close_curl (GstElement * element)
{
    GstUrlsrc *src;
    src = GST_URLSRC (element);
    g_return_if_fail (GST_IS_URLSRC (src));
    
    if (GST_FLAG_IS_SET(src,GST_URLSRC_OPEN)) 
    {
	   curl_easy_cleanup(src->curl);
	   src->curl=0; 
	   GST_FLAG_UNSET(src,GST_URLSRC_OPEN);
	   
    }
    return TRUE;
	
}

static gboolean gst_urlsrc_open_curl (GstElement * element)
{
    GstUrlsrc *src;
    src = GST_URLSRC (element);
    g_return_if_fail (GST_IS_URLSRC (src));
    
    if (!GST_FLAG_IS_SET(src,GST_URLSRC_OPEN)) 
    {
	   src->curl = curl_easy_init();
	   g_return_if_fail(src->curl);
	   
           curl_easy_setopt( src->curl, CURLOPT_URL, src->url );
           curl_easy_setopt( src->curl, CURLOPT_TIMEOUT, 10 );
           curl_easy_setopt( src->curl, CURLOPT_WRITEFUNCTION, 
			   WriteMemoryCallback );
	   //curl_easy_setopt( src->curl, CURLOPT_FORBID_REUSE, 1 );
           //curl_easy_setopt( src->curl, CURLOPT_TCP_NODELAY, 0 );
	   //curl_easy_setopt( src->curl, CURLOPT_BUFFERSIZE, 500000);
	   
	   GST_FLAG_SET(src,GST_URLSRC_OPEN);
	   src->sent_bytes=0;
	   
    }
    return TRUE;
	
}

static GstData *
gst_urlsrc_get (GstPad *pad)
{
    GstBuffer *buffer;
    GstUrlsrc *src;
    GstClockTime time_now;
    struct timeval tv;
    struct timezone tz;
       
    g_return_if_fail (GST_IS_PAD (pad));
    src = GST_URLSRC (GST_OBJECT_PARENT (pad));
    g_return_if_fail (GST_IS_URLSRC (src));
    
    buffer = gst_buffer_new_and_alloc (500000);
    g_return_if_fail (buffer != NULL);
    GST_BUFFER_SIZE(buffer) = 0;
   
    /* check for flush */
    if (src->need_flush) {
	    src->need_flush = FALSE;
	    GST_DEBUG_OBJECT (src, "sending flush");
	    return GST_DATA (gst_event_new_flush ());
    }
    
    /* check for seek */
    if (src->need_discont) {
	    GstEvent *event;
	    GST_DEBUG_OBJECT (src, "sending discont");
	    event = gst_event_new_discontinuous (
			    src->need_discont > 1, GST_FORMAT_UNDEFINED);
	    src->need_discont = 0;
	    gettimeofday( &(tv), &(tz) );
	    src->time_start=GST_TIMEVAL_TO_TIME(tv);
	    return GST_DATA (event);
    }
    

    if (GST_FLAG_IS_SET(src,GST_URLSRC_OPEN))
    {
        curl_easy_setopt( src->curl, CURLOPT_WRITEDATA, (void*) buffer );
	while ( curl_easy_perform( src->curl ) )
	{
	   GST_BUFFER_SIZE(buffer) = 0;
	   usleep(10000);
	   GST_INFO_OBJECT(src, "Attesa\n");
	}
    } else {
   	g_assert_not_reached();
    }

    gettimeofday( &(tv), &(tz) );
    time_now=GST_TIMEVAL_TO_TIME(tv);
    
    GST_BUFFER_TIMESTAMP (buffer) = time_now - src->time_start;
    
    //g_print("Timestamp %llu = %llu - %llu\n", GST_BUFFER_TIMESTAMP (buffer)  , time_now , src->time_start);
    
    return GST_DATA (buffer);
}

static void
gst_urlsrc_set_property (GObject *object, guint prop_id,
                                  const GValue *value, GParamSpec *pspec)
{
  GstUrlsrc *filter;

  g_return_if_fail (GST_IS_URLSRC (object));
  filter = GST_URLSRC (object);

  switch (prop_id)
  {
  case ARG_URL:
    filter->url = g_value_dup_string (value);
    GST_INFO_OBJECT( object,"%s\n",filter->url );
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    break;
  }
}

static void
gst_urlsrc_get_property (GObject *object, guint prop_id,
                                  GValue *value, GParamSpec *pspec)
{
  GstUrlsrc *filter;

  g_return_if_fail (GST_IS_URLSRC (object));
  filter = GST_URLSRC (object);

  switch (prop_id) {
  case ARG_URL:
    g_value_set_string (value, filter->url);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    break;
  }
}

static GstElementStateReturn
gst_urlsrc_change_state (GstElement * element)
{
  GstUrlsrc *src = GST_URLSRC (element);

  switch (GST_STATE_TRANSITION (element)) {
    case GST_STATE_NULL_TO_READY:
      break;
    case GST_STATE_READY_TO_NULL:
      break;
    case GST_STATE_READY_TO_PAUSED:
      if (!GST_FLAG_IS_SET (element, GST_URLSRC_OPEN)) {
        if (!gst_urlsrc_open_curl (GST_URLSRC (element)))
          return GST_STATE_FAILURE;
      }
      src->need_discont = 2;
      break;
    case GST_STATE_PAUSED_TO_READY:
      if (GST_FLAG_IS_SET (element, GST_URLSRC_OPEN))
        gst_urlsrc_close_curl (GST_URLSRC (element));
      break;
    default:
      break;
  }

  if (GST_ELEMENT_CLASS (parent_class)->change_state)
    return GST_ELEMENT_CLASS (parent_class)->change_state (element);

  return GST_STATE_SUCCESS;
}

/* entry point to initialize the plug-in
 * initialize the plug-in itself
 * register the element factories and pad templates
 * register the features
 */
static gboolean
plugin_init (GstPlugin *plugin)
{
  return gst_element_register (plugin, "urlsrc",
			       GST_RANK_NONE,
			       GST_TYPE_URLSRC);
}

static GstElementStateReturn gst_urlsrc_change_state (
						GstElement *element);

/* this is the structure that gst-register looks for
 * so keep the name plugin_desc, or you cannot get your plug-in registered */
GST_PLUGIN_DEFINE (
  GST_VERSION_MAJOR,
  GST_VERSION_MINOR,
  "urlsrc",
  "Template plugin",
  plugin_init,
  VERSION,
  "LGPL",
  "GStreamer",
  "http://gstreamer.net/"
)
