/* 
 * usbcam based SPCA5xx Image Capture Program for NetBSD, version 0.3
 * spca5shot - Simple image capture program.
 * Copyright (C) 2003,2005 Takafumi Mizuno <taka-qce at ls-a.jp>
 *
 * Portions of this program were modeled after or adapted from the
 * version 0.56 of the SPCA5xx video for linux (v4l) driver.
 * SPCA5xx version by Michel Xhaard <mxhaard _at_ users.sourceforge.net>
 * Based on :
 * SPCA50x version by Joel Crisp <cydergoth _at_ users.sourceforge.net>
 * OmniVision OV511 Camera-to-USB Bridge Driver
 * Copyright (c) 1999-2000 Mark W. McClelland
 * Kernel 2.6.x port Michel Xhaard && Reza Jelveh (feb 2004)
 *   and many people; see spca50x.c and ../../README and URL as follow;
 * http://spca50x.sourceforge.net/
 * http://mxhaard.free.fr/
 *
 * This program is free software; you can redistribute it and/or modify
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <sys/ioctl.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <libgen.h>	/* basename */
#include <stdarg.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <err.h>

#include <dev/usb/usb.h>

#include "linux_usbif.h"
#include "spca50x.h"
#include "spcadecoder.h"
#include "spca5shot.h"

#ifndef DEFHEIGHT
#define DEFHEIGHT LOCAL_Y
#endif

#ifndef DEFWIDTH
#define DEFWIDTH LOCAL_X
#endif

#define IFNUM 0
#define USEFRAMENUM	0

struct usb_device usbdev;
struct usb_spca50x *spcadev;

char dev[FILENAME_MAX];

unsigned char isobuf[MAX_FRAME_SIZE_PER_DESC];	/* one raw isoc packets data */

int errnum = 0;
static int infflag = 0;
static int jpegout = 0;
static int abflag = 0;
static int timflag = 0;
static int quietflag = 0;
static int bcflag = 0;	/* brightness/contrast */
static int stream;
#define	BFLAG	0x01
#define	CFLAG	0x02
#define	WFLAG	0x04
#define	HFLAG	0x08
#define	RFLAG	0x10
static int extime = EXPOSURE_TIMES;

static void outputppm(unsigned char *target, int x, int y, char *name);
static void outputraw(unsigned char *target, int length, char *name);

void usage(char *pname)
{
    fprintf (stderr,
	     "Usage: %s [-d device] [-s NxN] [-o file] [-b brightness] [-c contrast] [-e time] [-a] [-t] [-i] [-h] [-D level] [-p]\n"
	     " -d device    Specify the target device. Absolute path are accepted.\n"
	     " -s NxN       Specify size of the output image. (default:%dx%d)\n"
	     " -o file      Specify output file name instead of stdout.\n"
	     " -b brightness Specify the brightness. [0-65535]\n"
	     " -c contrast  Specify the contrast. [0-65535]\n"
	     " -C colour    Specify the colour. [0-65535]\n"
	     " -w whitenes  Specify the whitenes. [0-65535]\n"
	     " -H hue       Specify the hue. [0-65535]\n"
	     " -e times     Specify exposure times. (default:%d)\n"
	     " -a           Specify enable auto brightness with chipset support.\n"
	     " -t           Display frame rate after output a image.\n"
	     " -i           Display information of the camera without a output image.\n"
	     " -h           Display usage. (this message)\n"
	     " -D level     Specify the level of debug message. (0:default - 5:verbose)\n"
	     " -p           Specify dump %d isoc packets. (for debug)\n"
	     " -S           Stream images in a HTTP MJpeg stream.\n"
#ifdef __FreeBSD__
	     "Example: %s -d /dev/ugen1\n",
#else
	     "Example: %s -d /dev/ugen1.00\n",
#endif
	     (char*)basename(pname),
	     LOCAL_X, LOCAL_Y, EXPOSURE_TIMES, DUMP_PACKETS,
	     (char*)basename(pname));
    exit (1);
}

int ck_arg(char *arg, char *name)
{
char *s;

    s = rindex(arg, '/');

    if (s) {
	s++;
    } else {
	s = arg;
    }
    if (strcmp(s, name) == 0) {
	return 1;
    } else {
	return 0;
    }
}

int main(int argc, char *argv[])
{
    int c,optmp;
    int i = 0;
    int ret = 0;
    char *devname = NULL;
    struct timeval tv[EXPOSURE_TIMES];
    int ti = 0;	/* index of tv[] */
    struct video_mmap	vm;    
    struct video_picture vp;
    __u16 brightness = 0;
    __u16 contrast = 0;
    __u16 whiteness = 0;
    __u16 hue = 0;
    __u16 colour = 0;

    setlinebuf(stderr); setlinebuf(stdout);   

    vm.height = DEFHEIGHT;
    vm.width  = DEFWIDTH;

    if (ck_arg(argv[0], "spca5stream")) {
	stream = 1;
	jpegout = 1;
	quietflag = 1;
    } else {

    /* 
     * check option
     */
    while ((c = getopt(argc, argv, "ab:b:C:c:d:e:ijo:pqSs:tH:W:w:D:h")) != EOF) {
	switch (c) {
	case 'a':
	    abflag = 1;
	    break;
	case 'b':
	    bcflag |= BFLAG;
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 65535 ) {
		usage(argv[0]);		
	    }
	    brightness = (__u16)optmp;
	    break;
	case 'c':
	    bcflag |= CFLAG;
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 65535 ) {
		usage(argv[0]);		
	    }
	    contrast = (__u16)optmp;
	    break;
	case 'C':
	    bcflag |= RFLAG;
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 65535 ) {
		usage(argv[0]);		
	    }
	    colour = (__u16)optmp;
	    break;
	case 'W':
	case 'w':
	    bcflag |= WFLAG;
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 65535 ) {
		usage(argv[0]);		
	    }
	    whiteness = (__u16)optmp;
	    break;

	case 'H':
	    bcflag |= HFLAG;
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 65535 ) {
		usage(argv[0]);		
	    }
	    hue = (__u16)optmp;
	    break;

	case 'd':
	    devname = optarg;
	    break;
	case 'e':
	    optmp = atoi(optarg);
	    if ( optmp < 0 ) {
		usage(argv[0]);		
	    }
	    extime = optmp;
	    break;
	case 'i':
	    infflag = 1;
	    break;
	case 'S':
	    stream = 1;
	    /* fall through */
	    /* jpeg output assumed. */
	case 'j':
	    jpegout = 1;
	    break;
	case 'o':
	    filename = optarg;
	    break;
	case 'p':
	    dumpflag = 1;
	    break;
	case 's':
	    if ( sscanf(optarg, "%dx%d", &vm.width, &vm.height) != 2 ) {
		usage(argv[0]);
	    }
	    break;
	case 't':
	    timflag = 1;
	    break;
	case 'q':
	    quietflag = 1;
	    break;
	case 'D':
	    optmp = atoi(optarg);
	    if ( optmp < 0 || optmp > 5 ) {
		usage(argv[0]);		
	    }
	    debug = optmp;
	    break;
	case 'h':
	case '?':
	default:
	    usage(argv[0]);
	    break;
	}
    }

    if ( argc != optind ) {
	usage(argv[0]);
    }
    }


    if ( devname == NULL ) {
	/* search camera */
	for ( i = 0; i < 15; ++i ) {
#ifdef __FreeBSD__
	    sprintf(dev, "/dev/ugen%d", i);
#else
	    sprintf(dev, "/dev/ugen%d.00", i);
#endif
	    if ( (usbdev.fd = open(dev, O_RDWR)) < 0 ) {
		continue;
	    }
	    if ( (spcadev = (struct usb_spca50x *)spca50x_probe(&usbdev, IFNUM)) == NULL ) {
		close(usbdev.fd);
		usbdev.fd = -1;
		continue;
	    } else {
		break;
	    }
	}
    } else {
	if ( (usbdev.fd = open(devname, O_RDWR)) < 0 ) {
	    perror(devname);
	} else {
	    if ( (spcadev = (struct usb_spca50x *)spca50x_probe(&usbdev, IFNUM)) == NULL ) {
		close(usbdev.fd);
		usbdev.fd = -1;
	    } else {
		(void)strcpy(dev, devname);
	    }
	}
    }

    if ( usbdev.fd < 0 ) {
	fprintf(stderr, "Not found SPCA50x based usb camera, or Permission denied\n");
	exit(1);
    }

    if ( spca50x_open(spcadev) != 0 ) {
	ret = 1;
	goto OPENERR;
    }

    /* get/set picture param */
    spca50x_do_ioctl(spcadev, VIDIOCGPICT, &vp);
    if ( bcflag != 0 ) {
	if ( (bcflag & BFLAG) != 0 ) {
	    vp.brightness = brightness;
	}
	if ( (bcflag & CFLAG) != 0 ) {
	    vp.contrast   = contrast;
	}
	if ( (bcflag & WFLAG) != 0 ) {
	    vp.whiteness   = whiteness;
	}
	if ( (bcflag & HFLAG) != 0 ) {
	    vp.hue   = hue;
	}
	if ( (bcflag & RFLAG) != 0 ) {
	    vp.colour   = colour;
	}
	spca50x_do_ioctl(spcadev, VIDIOCSPICT, &vp);
    }

    if ( infflag != 0 ) {
	displayinfo(spcadev);
	goto DO_CLOSE;
    }

    vm.frame = USEFRAMENUM;
    if (jpegout == 1) {
	vm.format = VIDEO_PALETTE_JPEG;
    } else {
	vm.format = VIDEO_PALETTE_RGB24;
    }

    if ( spca50x_do_ioctl(spcadev, VIDIOCMCAPTURE, &vm) != 0 ) {
	ret = 1;
	goto MODEINIERR;
    }

    /* open endpoint  */
#ifdef __FreeBSD__
    sprintf(usbdev.epdevname, "%s.%d", dev, spcadev->endpoint_address);
#else
    sprintf(usbdev.epdevname, "%s.%02d", strtok(dev,"."), spcadev->endpoint_address);
#endif
    if ((usbdev.efd = open(usbdev.epdevname, O_RDONLY)) < 0) {
	perror(usbdev.epdevname);
	ret = 2;
	goto EPOPENERR;
    } 

    memset((void *)isobuf, 0, MAX_FRAME_SIZE_PER_DESC);
    if ( timflag != 0 ) {
	for ( ti = 1; ti < EXPOSURE_TIMES; ti++) {
	    tv[ti].tv_sec  = -1L;
	    tv[ti].tv_usec = -1L;
	}
	ti = 0;
	gettimeofday(&tv[ti], NULL); /* tv[0] */
    }



    i=0;
    c=0;
    if (stream) {
	// greet our client;
	printf("HTTP/1.0 200 OK\r\n"
	    "Server: spca5 Web Server/1.0\r\n"
	    "Content-Type: multipart/x-mixed-replace;boundary=--IPCamBoundary--\r\n"
	    "MIME-version: 1.0\r\n"
	    "Pragma: no-cache\r\n"
	    "Cache-Control: no-cache\r\n"
	    "Expires: 01 Jan 1970 00:00:00 GMT\r\n"
	    "\r\n");

	while(1) {
	    if ( spcadev->bridge == BRIDGE_TV8532 ) {
		ret = tv8532_grab(spcadev);
	    } else {
		ret = spca_grab(spcadev);
	    }
	    if ( abflag != 0 ) {
		auto_bh((void *)spcadev);
	    }
	    if ( ret < 0 ) {
		i++;
	    } else {
		i=0;
	    }
	    /* break after 10 errors in a row */
	    if ( i > 10 ) {
		ret=1;
	    break;
	    }
	    printf( "--IPCamBoundary--\r\n"
		"ETag: image_%06d\r\n"
		"Content-Type: image/jpeg\r\n"
		"Content-Length: %d\r\n"
		"\r\n",
		c++, 
		(int) spcadev->frame[USEFRAMENUM].scanlength );

	    ret = fwrite( spcadev->frame[USEFRAMENUM].data,
		 spcadev->frame[USEFRAMENUM].scanlength, 1, stdout);

	    if (ret < 1) {
		break;
		ret=1;
	    }
	} /* while 1 */
    } else {
	for ( i = extime; i > 0; i-- ) {
	    if ( spcadev->bridge == BRIDGE_TV8532 ) {
		ret = tv8532_grab(spcadev);
	    } else {
		ret = spca_grab(spcadev);
	    }
	    if ( timflag != 0 ) {
		ti = (ti + 1) % EXPOSURE_TIMES;
		gettimeofday(&tv[ti], NULL);
	    }
	    if ( abflag != 0 ) {
		auto_bh((void *)spcadev);
	    }
	    if ( ret < 0 ) {
		ret=1;
		break;
	    }
	    if (!quietflag) 
		fprintf(stderr, "exposure time: last %d    \r", i);

	    if (!quietflag) 
	    fprintf(stderr, "brightness %d, colour %d, contrast %d, hue %d, whiteness %d\n",
		spcadev->brightness, spcadev->colour, spcadev->contrast, spcadev->hue,
		spcadev->whiteness);
	}

	if ( i == 0 ) {

	    if (jpegout == 1) {
		outputraw(spcadev->frame[USEFRAMENUM].data, (int) spcadev->frame[USEFRAMENUM].scanlength,
		    filename);
	    } else {
		outputppm(spcadev->frame[USEFRAMENUM].data,
		      spcadev->frame[USEFRAMENUM].width, spcadev->frame[USEFRAMENUM].height,
		      filename);
	    }

       
	ret = 0;
	if ( timflag != 0 ) {
	    if ( extime < EXPOSURE_TIMES ) {
		fprintf(stderr, "frame rate: can not calculate. at lease exposure times must be more than %d, check -e option.\n", EXPOSURE_TIMES);
	    } else {
		fprintf(stderr, "frame rate: %.2f(fps)\n", calc_framerate(tv, ti, NULL));
	    }
	}

	} else {
	    ret = 1;
	}
    }

    close(usbdev.efd);

DO_CLOSE:
EPOPENERR:
MODEINIERR:
    spca50x_close(spcadev);
OPENERR:
    spca50x_disconnect(&usbdev, (void *)spcadev);
    close(usbdev.fd);
    if (!quietflag)
	fprintf(stderr, "\nDone.\n");

    return ret;
}


static void outputraw(unsigned char *target, int length, char *out)
{
    int i;
    FILE *fid;

    if ( target == NULL ) return;

    if ( out != NULL ) {
	if ( (fid = fopen(out, "wb")) == NULL ) {
	    perror("fopen");
	    return;
	}
    } else {
	/* stdout */
	fid = stdout;
    }

    i = fwrite(target, length, 1, fid);
    if (i < 1) {
	fprintf(stderr, "wrote %d instead of %d", i, length);
	perror("fwrite");
	return;
    }

    if ( out != NULL ) {
	fclose(fid);
    }
}

static void outputppm(unsigned char *target, int x, int y, char *out)
{
    int i;
    FILE *fid;

    if ( target == NULL ) return;

    if ( out != NULL ) {
	if ( (fid = fopen(out, "wb")) == NULL ) {
	    perror("fopen");
	    return;
	}
    } else {
	/* stdout */
	fid = stdout;
    }

    fprintf(fid, "P6\n");
    fprintf(fid, "%d %d 255 ", x, y);
    for ( i = 0; i < (x*y*3); i+=3 ) {
	fprintf(fid, "%c%c%c",
		target[i],
		target[i+1],
		target[i+2]);
    }

    if ( out != NULL ) {
	fclose(fid);
    }

    return;
}

